/*
 * Copyright (c) 2011-2014 Intel Corporation. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: $
 */

#include <sys/inotify.h>
#include <sys/wait.h>
#include <pthread.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <syslog.h>
#include <limits.h>
#include <signal.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>

#define DEV_DIR		"/dev"
#define EVENT_SIZE_MAX	(sizeof(struct inotify_event) + NAME_MAX + 1)

int nthreads = 4;

void *ibpd_thread(void *devname)
{
	int fd, ret;

	if ((fd = open(devname, O_WRONLY)) != -1) {
		ret = write(fd, NULL, 0);
		close(fd);
	}

	return NULL;
}

int ibpd(char *devname, int nthreads)
{
	pthread_t *thread;
	int i;

	thread = calloc(nthreads, sizeof *thread);
	if (!thread)
		return ENOMEM;

	syslog(LOG_INFO, "pid %d %s started %d thread%s\n",
	       getpid(), devname, nthreads, nthreads == 1 ? "" : "s");

	for (i = 0; i < nthreads; i++)
		pthread_create(&thread[i], NULL, ibpd_thread, devname);

	for (i = 0; i < nthreads; i++)
		pthread_join(thread[i], NULL);

	syslog(LOG_INFO, "pid %d %s stopped %d thread%s\n",
	       getpid(), devname, nthreads, nthreads == 1 ? "" : "s");

	free(thread);
	return 0;
}

void reap_ibpd(int sig)
{
	wait(NULL);
}

void start_ibpd(char *name)
{
	char *devname;

	if (!!fork())
		return;

	devname = malloc(strlen(DEV_DIR) + strlen(name) + 2);
	if (!devname) {
		syslog(LOG_ERR, "pid %d malloc failed\n", getpid());
		exit(EXIT_FAILURE);
	}
	sprintf(devname, "%s/%s", DEV_DIR, name);

	ibpd(devname, nthreads);

	free(devname);
	exit(EXIT_SUCCESS);
}

void parse_events(void *buf, size_t size)
{
	struct inotify_event *event;
	unsigned int node;
	int i;

	for (i = 0; i < size; i += sizeof(struct inotify_event) + event->len) {
		event = (struct inotify_event *)(buf + i);
		if (event->len && sscanf(event->name, "ibp%u", &node) == 1)
			start_ibpd(event->name);
	}
}

int main(int argc, char **argv)
{
	int fd;
	int wd;
	int opt;
	int ret;
	void *buf;

	while ((opt = getopt(argc, argv, "t:")) != -1) {
		switch (opt) {
		case 't':
			nthreads = atoi(optarg);
			break;
		default:
			syslog(LOG_ERR, "Usage: %s [-t nthreads]\n", argv[0]);
			return EINVAL;
		}
	}

	if (nthreads < 1 || nthreads > sysconf(_SC_NPROCESSORS_CONF)) {
		syslog(LOG_ERR,
		       "nthreads %d invalid, expected 1 <= nthreads <= nCPUs\n",
		       nthreads);
		return EINVAL;
	}

	fd = inotify_init();
	if (fd < 0) {
		syslog(LOG_ERR, "inotify_init: %s\n", strerror(errno));
		goto err0;
	}

	wd = inotify_add_watch(fd, DEV_DIR, IN_CREATE);
	if (wd < 0) {
		syslog(LOG_ERR, "inotify_add_watch: %s\n", strerror(errno));
		goto err1;
	}

	buf = malloc(EVENT_SIZE_MAX);
	if (!buf) {
		syslog(LOG_ERR, "event buf malloc failed\n");
		goto err2;
	}

	signal(SIGCHLD, reap_ibpd);

	for (;;) {
		ret = read(fd, buf, EVENT_SIZE_MAX);
		if (ret < 0) {
			/* Ignore EINTR (from child termination on some OSes) */
			if (errno == EINTR)
				continue;
			syslog(LOG_ERR, "read: %s\n", strerror(errno));
			break;
		}

		parse_events(buf, ret);
	}

	free(buf);
err2:
	inotify_rm_watch(fd, wd);
err1:
	close(fd);
err0:
	return errno;
}
